#!/bin/sh -e

# Function for asking the user if everything is ok
continuep () {
	while read -p "Does this look reasonable (y/n)? " CONTINUE; do
		case "${CONTINUE}" in
		y*)
			return 0
			;;
		n*)
			return 1
			;;
		esac
	done
}

# Print user-friendly progress statistics
# Borrowed from portsnap.
fetch_progress() {
        LNC=0
        while read x; do
                LNC=$(($LNC + 1))
                if [ $(($LNC % 10)) = 0 ]; then
                        echo -n $LNC
                elif [ $(($LNC % 2)) = 0 ]; then
                        echo -n .
                fi
        done
        echo -n " "
}

# Function for installing directories, files, and symlinks as specified
# in $1.  Targets of hard links must be lexicographically earlier than
# the paths which link to them.
install_from_index () {
	# First pass: Do everything apart from setting file flags.  We
	# can't set flags yet, because schg inhibits hard linking.
	awk -F \| '{ print $1,$2,$4,$5,$6,$7,$8,$3 }' $1 |
	    sort |
	    while read FPATH TYPE OWNER GROUP PERM FLAGS HASH LINK; do
		case ${TYPE} in
		d)
			# Create a directory
			install -d -o ${OWNER} -g ${GROUP}		\
			    -m ${PERM} ${FPATH}
			;;
		f)
			if [ -z "${LINK}" ]; then
				# Create a file, without setting flags.
				install -S -o ${OWNER} -g ${GROUP}	\
				    -m ${PERM} files/${HASH} ${FPATH}
			else
				# Create a hard link.
				ln -f ${LINK} ${FPATH}
			fi
			;;
		L)
			# Create a symlink
			ln -sfh ${HASH} ${FPATH}
			;;
		esac
	    done

	# Perform a second pass, adding file flags.
	awk -F \| '{ print $1,$2,$4,$5,$6,$7,$8,$3 }' $1 |
	    while read FPATH TYPE OWNER GROUP PERM FLAGS HASH LINK; do
		if [ ${TYPE} = "f" ] &&
		    ! [ ${FLAGS} = "0" ]; then
			chflags ${FLAGS} ${FPATH}
		fi
	    done
}

echo -n "Examining system..."

# Make sure that we're starting from FreeBSD 6.0.
case `uname -r` in
6.0-*)
	;;
*)
	echo <<EOF

You are not running FreeBSD 6.0.  This script is only designed to upgrade
FreeBSD 6.0 systems to FreeBSD 6.1.
EOF
	exit 1
	;;
esac

# Make sure that we're running on i386.
case `uname -p` in
i386)
	;;
*)
	echo <<EOF

You are not running FreeBSD on an i386 system.  This script is only 
designed to upgrade i386 systems.
EOF
	exit 1
	;;
esac

# Figure out which kernel we should be installing.
case `uname -i` in
GENERIC)
	echo 'kernel|generic' > comp.present
	KCONF=GENERIC
	;;
SMP-GENERIC)
	echo 'kernel|smp' > comp.present
	KCONF=SMP
	;;
*)
	cat <<EOF

You are not running a GENERIC or SMP kernel.  Since those are the only
kernels distributed with FreeBSD 6.1-RELEASE, this script cannot perform
an upgrade automatically.
EOF
	exit 1
	;;
esac

# Make sure the kernel is loaded from /boot/kernel/kernel, since that's
# where we're going to end up putting the new kernel.
if ! [ `sysctl -n kern.bootfile` = "/boot/kernel/kernel" ]; then
	cat <<EOF

The currently running kernel was loaded from `sysctl -n kern.bootfile`.
This probably means that you've set the "kernel" or "bootfile" variables
in /boot/loader.conf.  This script ends up installing a FreeBSD 6.1 kernel,
to /boot/kernel/kernel, which would likely make this system unbootable; as
a result, it will not proceed.
EOF
	exit 1
fi

# Make sure securelevel <= 0.  Otherwise we can't get rid of schg files.
if [ `sysctl -n kern.securelevel` -gt 0 ]; then
	cat <<EOF

The system securelevel is currently set to `sysctl -n kern.securelevel`.  This
is greater than zero, which means that the system immutable (schg) flag
cannot be removed from files.  Reboot with a lower securelevel before
re-running this script.
EOF
	exit 1
fi

# For each component which can be installed, count how many files are
# present on the system; if a component is more than 50% present, then
# assume it's supposed to be installed (and put it into the list of
# components to install from the new release).
cut -f 1-3 -d '|' 6.0-index |
    tr '|' ' ' |
    while read COMP SCOMP FPATH; do
	if [ -e ${FPATH} ]; then
		echo "${COMP}|${SCOMP}"
	fi
    done |
    uniq -c |
    sed -E 's,^ +,,' > compfreq.present
cut -f 1-2 -d '|' 6.0-index |
    uniq -c |
    sed -E 's,^ +,,' > compfreq.all
join -1 2 -2 2 compfreq.all compfreq.present |
    while read SCOMP TOTAL PRESENT; do
	if [ ${PRESENT} -gt `expr ${TOTAL} / 2` ]; then
		echo ${SCOMP}
	fi
    done >> comp.present
rm compfreq.all compfreq.present

# Also generate the list of components which are *not* present.
cut -f 1-2 -d '|' 6.1-index |
    uniq |
    comm -23 - comp.present > comp.absent

# Update the user on our progress, and ask him to sanity-check.
echo " done."
echo
echo "The following components of FreeBSD seem to be installed:"
fmt -72 < comp.present
echo
echo "The following components of FreeBSD do not seem to be installed:"
fmt -72 < comp.absent
echo
continuep
echo
echo -n "Examining system (this will take a bit longer)..."

# Pull out the subset of the index of files from the new release which
# corresponds to the components which we want to install.
while read SCOMP; do
	look ${SCOMP} 6.1-index
done < comp.present |
    cut -f 3- -d '|' |
    sort -u > 6.1-subindex

# Similarly, generate the subindex containing the components which we
# think should be installed; but then filter it to remove anything which
# isn't actually there.
while read SCOMP; do
	look ${SCOMP} 6.0-index || true
done < comp.present |
    cut -f 3- -d '|' |
    sort -u |
    tr '|' ' ' |
    while read FPATH TYPE HASH; do
	if [ -e ${FPATH} ]; then
		echo "${FPATH}|${TYPE}|${HASH}"
	fi
    done > 6.0-subindex

# We don't need the component lists any more.
rm comp.absent comp.present

# Make sure all the installed files|symlinks|directories are, in fact,
# files|symlinks|directories.
tr '|' ' ' < 6.0-subindex |
    while read FPATH TYPE HASH; do
	case ${TYPE} in
	d)
		if ! [ -d ${FPATH} ]; then
			echo "${FPATH} should be a directory, but isn't!"
			exit 1
		fi
		;;
	f)
		if ! [ -f ${FPATH} ]; then
			echo "${FPATH} should be a file, but isn't!"
			exit 1
		fi
		;;
	L)
		if ! [ -L ${FPATH} ]; then
			echo "${FPATH} should be a symlink, but isn't!"
			exit 1
		fi
		;;
	esac
    done

# Check return code of previous pipeline
[ $? = 0 ]

# For each installed regular file from the old release, compute the
# hash of the file actually present on disk.  We're going to use these
# to determine which files have been modified since they were installed.
#
# XXX We don't detect if a symlink has been changed to point somewhere
# XXX else.  I doubt this will be a problem for anyone.
tr '|' ' ' < 6.0-subindex |
    while read FPATH TYPE HASH; do
	if [ ${TYPE} = "f" ]; then
		echo -n "${FPATH}|${TYPE}|"
		sha256 -q ${FPATH}
	fi
    done |
    uniq > here-index

# Generate list of files which don't match the released versions
comm -23 here-index 6.0-subindex |
    cut -f 1 -d '|' |
    sort > modified.files

# Split the list of modified files into
# 1. Configuration files which might need changed merged,
# 2. Data files (e.g., logs) which should be left untouched, and
# 3. Other modifications, probably accidental, which should be
#    overwritten by the version in the new release.
grep -E '^/etc/|^/var/named/etc/' modified.files > modified-merge
comm -23 modified.files modified-merge |
    grep '^/var/' > modified-keep
comm -23 modified.files modified-merge |
    grep -v '^/var/' > modified-overwrite

# Some of the configuration files which theoretically could need to
# have changes merged in fact haven't changed between releases.  Find
# them and put them into the 'modified-keep' list instead.
sort -k 1,1 -t '|' 6.0-subindex |
    join -t '|' modified-merge - > modified-merge-olds
sort -k 1,1 -t '|' 6.1-subindex |
    join -t '|' modified-merge -  |
    join -t '|' -o 1.1,1.3,2.8 modified-merge-olds - |
    tr '|' ' ' |
    while read FPATH OHASH NHASH; do
	if [ ${OHASH} = ${NHASH} ]; then
		echo ${FPATH}
	fi
    done |
    sort - modified-keep > modified-keep.tmp
mv modified-keep.tmp modified-keep

# ... and remove them from the modified-merge list.
sort modified-keep modified-overwrite |
    comm -23 modified.files - > modified-merge

# XXX Certain configuration files need to be handled specially:
# XXX 1. /etc/master.passwd, /etc/passwd, /etc/*pwd.db, /etc/group,
# XXX    since these need to be updated before files are installed
# XXX    with newly added owners.
# XXX 2. /etc/login.conf, /etc/login.conf.db,
# XXX    since the database is compiled from the configuration file.
#
# Fortunately, out of this list only /etc/group has changed between
# FreeBSD 6.0 and FreeBSD 6.1.

# Update the user on our status, and ask him to sanity-check.
echo " done."
echo
cat <<EOF
The following files from FreeBSD 6.0 have been modified since they were
installed, but will be deleted or overwritten by new versions:
EOF
fmt -72 < modified-overwrite |
    more
echo
cat <<EOF
The following files from FreeBSD 6.0 have been modified since they were
installed, and will not be touched:
EOF
fmt -72 < modified-keep
echo
cat <<EOF
The following files from FreeBSD 6.0 have been modified since they were
installed, and the changes in FreeBSD 6.1 will be merged into the
existing files:
EOF
fmt -72 < modified-merge
echo
continuep
echo

# Copy files from the world which have not been modified since the release.
echo -n "Preparing to fetch files..."
mkdir -p files
sort -k 1,1 -t '|' here-index |
    join -t '|' -v 2 modified.files - > here-unmodified
tr '|' ' ' < here-unmodified|
    while read FPATH F HASH; do
	cp ${FPATH} files/${HASH}
    done

# Identify which patches we need to download.
grep '|f|' 6.1-subindex |
    sort -k 1,1 -t '|' |
    join -t '|' -o 1.3,2.8 here-unmodified - |
    tr '|' ' ' |
    while read OHASH NHASH; do
	if ! [ ${OHASH} = ${NHASH} ]; then
		echo "${OHASH}|${NHASH}"
	fi
    done |
    sort -u > bp.list

echo " done."

# Download patches.
echo -n "Fetching `wc -l < bp.list | tr -d ' '` patches"
mkdir -p bp
lam -s '/6.0-to-6.1/bp/' bp.list |
    tr '|' '-' |
    (
	cd bp/ &&
	    xargs /usr/libexec/phttpget upgrade.daemonology.net
    ) 2>&1 | fetch_progress
echo "done."

# Apply patches.
echo -n "Applying patches..."
mkdir -p tmp/
tr '|' ' ' < bp.list |
    while read OHASH NHASH; do
	bspatch files/${OHASH} tmp/${NHASH} bp/${OHASH}-${NHASH}
	if [ ${NHASH} = `sha256 -q tmp/${NHASH}` ]; then
		mv tmp/${NHASH} files/${NHASH}
	fi
    done
echo " done."

# Figure out which files we need.  We need a file if it is part of
# 6.1-subindex but *not* part of the modified-keep list; and we also
# need files from 6.0-subindex which are part of the modified-merge
# list (for comparison purposes).

grep '|f|' 6.0-subindex |
    sort -t '|' -k 1,1 |
    join -t '|' modified-merge - |
    cut -f 3 -d '|' > f.list.tmp
[ $? = 0 ]		# Check return code; sh -e doesn't do pipelines.
grep '|f|' 6.1-subindex |
    sort -t '|' -k 1,1 |
    join -t '|' -v 2 modified-keep - |
    cut -f 8 -d '|' |
    sort -u - f.list.tmp > f.list
[ $? = 0 ]		# Check return code; sh -e doesn't do pipelines.
rm f.list.tmp

# Figure out which files we need and don't yet have.
ls files |
    comm -13 - f.list > f.needed

# Fetch them.
echo -n "Fetching `wc -l < f.needed | tr -d ' '` files"
lam -s '/6.0-to-6.1/f/' f.needed -s '.gz' |
    (
	cd tmp/ &&
	    xargs /usr/libexec/phttpget upgrade.daemonology.net
    ) 2>&1 | fetch_progress
echo "done."

# Decompress and verify them.
echo -n "Decompressing and verifying..."
while read HASH; do
	gunzip tmp/${HASH}.gz
	if [ ${HASH} = `sha256 -q tmp/${HASH}` ]; then
		mv tmp/${HASH} files/${HASH}
	else
		echo "Downloaded file is corrupt: ${HASH}.gz"
		exit 1
	fi
done < f.needed
echo " done."

# Clean up no-longer-needed bits
rm bp.list f.list f.needed
rm -r bp tmp

# Create staging area for merging configuration files
while read F; do
	D=`dirname ${F}`
	mkdir -p merge/old${D} merge/6.0${D} merge/6.1${D} merge/new${D}
	cp ${F} merge/old${F}
done < modified-merge

# Copy in the versions from 6.0
grep '|f|' 6.0-subindex |
    cut -f 1,3 -d '|' |
    sort -k 1,1 -t '|' |
    join -t '|' modified-merge - |
    tr '|' ' ' |
    while read F HASH; do
	cp files/${HASH} merge/6.0${F}
    done

# Copy in the versions from 6.1
grep '|f|' 6.1-subindex |
    cut -f 1,8 -d '|' |
    sort -k 1,1 -t '|' |
    join -t '|' modified-merge - |
    tr '|' ' ' |
    while read F HASH; do
	cp files/${HASH} merge/6.1${F}
    done

# Attempt to automatically merge changes
echo -n "Attempting to automatically merge configuration files..."
: > failed.merges
while read F; do
	# Treat /etc/group specially -- it is likely to have conflicts,
	# but we know how to handle them.
	case ${F} in
	/etc/group)
		cp merge/old${F} merge/new${F}
		echo "audit:*:77:" >> merge/new${F}
		;;
	*)
		if ! merge -p -L "current version" -L "FreeBSD 6.0"	\
		    -L "FreeBSD 6.1" merge/old${F}			\
		    merge/6.0${F} merge/6.1${F}				\
		    > merge/new${F} 2>/dev/null; then
			echo ${F} >> failed.merges
		fi
		;;
	esac
done < modified-merge
echo " done."

# Get the user to handle any files which didn't merge correctly.
while read F; do
	cat <<EOF

The following configuration file has changed which could not be
automatically merged into the version in FreeBSD 6.1: ${F}
Press Enter to edit this file in ${EDITOR} and resolve the conflicts
manually...
EOF
	read dummy < /dev/tty
	${EDITOR} `pwd`/merge/new${F} < /dev/tty
done < failed.merges
rm failed.merges

# Ask the user to confirm that he likes the merged configuration files.
while read F; do
	# If the merged version is the old version, skip.
	if cmp -s merge/old${F} merge/new${F}; then
		continue
	fi
	cat <<EOF

The following changes, which occurred between FreeBSD 6.0 and FreeBSD
6.1, have been merged into ${F}:
EOF
	diff -U 5 merge/old${F} merge/new${F} || true
	continuep < /dev/tty
done < modified-merge
echo

# Generate the final index of what to install.  We want to install:
# 1. Everything in 6.1-subindex EXCEPT the files listed in
#    modified-keep and modified-merge, and
# 2. The versions of files listed in modified-merge which we have
#    constructed (with some help from the user).
cat 6.1-subindex |
    sort -k 1,1 -t '|' > new-index.tmp
cat modified-merge modified-keep |
    sort -u |
    join -t '|' -v 2 - new-index.tmp > new-index.tmp2
while read F; do
	look "${F}|f|" new-index.tmp |
	    cut -f 1-7 -d '|' |
	    tr '\n' '|'
	sha256 -q merge/new${F}
	cp merge/new${F} files/`sha256 -q merge/new${F}`
done < modified-merge |
    sort -k 1,1 -t '|' - new-index.tmp2 > new-index
rm new-index.tmp new-index.tmp2

# Generate a list of files from the old release which can be
# overwritten or, if they are not present in the new release,
# deleted.  This is everything listed in 6.0-subindex which is
# not in modified-keep.
sort -t '|' -k 1,1 6.0-subindex |
    join -t '|' -v 1 - modified-keep |
    cut -f 1-2 -d '|' |
    uniq > old-index

# Make sure there isn't anything where we're about to install
# the new kernel (into /boot/SMP/ or /boot/GENERIC/) prior to
# renaming it to /boot/kernel/.
if [ -e "/boot/${KCONF}" ]; then
	echo -n "Moving /boot/${KCONF} to /boot/${KCONF}.old..."
	mv /boot/${KCONF} /boot/${KCONF}.old
	echo " done."
fi

# Install the new kernel
echo -n "Installing new kernel into /boot/${KCONF}..."
grep "^/boot/${KCONF}" new-index > new-index-kern
install_from_index new-index-kern
kldxref /boot/${KCONF}/
echo " done."

# Move the old kernel to /boot/kernel.old
if [ -e "/boot/kernel.old" ]; then
	echo -n "Deleting /boot/kernel.old..."
	chflags -R noschg /boot/kernel.old
	rm -rf /boot/kernel.old
	echo " done."
fi
echo -n "Moving /boot/kernel to /boot/kernel.old..."
mv /boot/kernel /boot/kernel.old
echo " done."

# And move the new kernel into place.
echo -n "Moving /boot/${KCONF} to /boot/kernel..."
mv /boot/${KCONF} /boot/kernel
echo " done."

# Remove the immutable flag on everything from the old release.
echo -n "Removing schg flag from existing files..."
grep -v '^/boot/kernel' old-index |
    cut -f 1 -d '|' |
    xargs chflags noschg
echo " done."

# Install everything non-kernel.
echo -n "Installing new non-kernel files..."
grep -v "^/boot/${KCONF}" new-index > new-index-nonkern
install_from_index new-index-nonkern
echo " done."

# Finally, figure out which bits from the old release should be
# cleaned up.  This is anything installed by the old release,
# except the files listed in modified-keep, which is not installed
# by the new release.
# In addition, we exclude /boot/kernel, since that is extracted
# under a different name.
# XXX For some reason, /usr/lib/liblwres.so had its version bumped.
# XXX Leave behind the old version just in case.
echo -n "Removing left-over files from FreeBSD 6.0..."
sort -t '|' -k 1,1 new-index |
    cut -f 1-2 -d '|' |
    join -t '|' -v 1 old-index - |
    grep -v '^/boot/kernel' |
    grep -v '/usr/lib/liblwres.so.3' |
    tr '|' ' ' |
    sort -r |
    while read FPATH TYPE; do
	if [ ${TYPE} = "d" ]; then
		rmdir ${FPATH}
	else
		rm ${FPATH}
	fi
    done
echo " done."

echo "To start running FreeBSD 6.1, reboot."
