#!/bin/sh -e

# Copyright (c) 2013 Colin Percival
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.
#
# $FreeBSD: user/cperciva/pkesh/pkesh.sh 257606 2013-11-04 05:18:19Z cperciva $

usage () {
	echo "usage: pkesh gen priv.key pub.key"
	echo "usage: pkesh enc pub.key in out"
	echo "usage: pkesh dec priv.key in out"
	exit 1
}

# gen priv.key pub.key
gen () {
	# Generate the key
	openssl genrsa -out "$D/rsakey" -f4 2048 2>/dev/null

	# Write out private and public parts
	cat "$D/rsakey" > "$1"
	openssl rsa -in "$D/rsakey" -pubout > "$2" 2>/dev/null
}

# enc pub.key in out
enc () {
	# Generate a random 256-bit AES key
	openssl rand 32 > "$D/aeskey"

	# Generate a random 128-bit IV
	openssl rand 16 > "$D/aesIV"

	# Generate the encrypted data
	KEY=`od -An -v -t x1 < "$D/aeskey" | tr -Cd '0-9a-fA-F'`
	IV=`od -An -v -t x1 < "$D/aesIV" | tr -Cd '0-9a-fA-F'`
	openssl enc -aes-256-cbc -K $KEY -iv $IV < "$2" > "$D/encdata"

	# Compute the SHA256 hash of the encrypted data
	openssl dgst -sha256 -binary "$D/encdata" > "$D/hash"

	# Generate the header
	cat "$D/aeskey" "$D/aesIV" "$D/hash" > "$D/header"

	# Generate the encrypted header
	openssl rsautl -inkey "$1" -pubin -encrypt -oaep \
	    < "$D/header" > "$D/encheader"

	# Generate the entire encrypted message
	cat "$D/encheader" "$D/encdata" | openssl enc -base64 > "$3"
}

# dec priv.key in out
dec () {
	# Base-64 decode the encrypted message
	openssl enc -d -base64 < "$2" > "$D/encmessage"

	# Make sure the message is long enough
	if [ `wc -c < "$D/encmessage"` -lt 256 ]; then
		echo "Message is corrupt or truncated" >/dev/stderr
		exit 1
	fi

	# Decrypt the header
	dd if="$D/encmessage" bs=256 count=1 of="$D/encheader" 2>/dev/null
	openssl rsautl -inkey "$1" -decrypt -oaep < "$D/encheader" > "$D/header"

	# Make sure the header is the right size
	if [ `wc -c < "$D/header"` -ne 80 ]; then
		echo "Message is corrupt" >/dev/stderr
		exit 1
	fi

	# Split header into components
	dd if="$D/header" bs=1 count=32 of="$D/aeskey" 2>/dev/null
	dd if="$D/header" bs=1 skip=32 count=16 of="$D/aesIV" 2>/dev/null
	dd if="$D/header" bs=1 skip=48 count=32 of="$D/hash" 2>/dev/null

	# Verify the encrypted data hash
	dd if="$D/encmessage" bs=256 skip=1 2>/dev/null |
	    openssl dgst -sha256 -binary > "$D/encmessage.hash"
	if ! cmp -s "$D/hash" "$D/encmessage.hash"; then
		echo "Message is corrupt or truncated" >/dev/stderr
		exit 1
	fi

	# Decrypt the message
	KEY=`od -An -v -t x1 < "$D/aeskey" | tr -Cd '0-9a-fA-F'`
	IV=`od -An -v -t x1 < "$D/aesIV" | tr -Cd '0-9a-fA-F'`
	dd if="$D/encmessage" bs=256 skip=1 2>/dev/null |
	    openssl enc -d -aes-256-cbc -K $KEY -iv $IV > "$3"
}

# Get operation type
if [ $# -lt 1 ]; then
	usage
fi
OP="$1"
shift

# Check operation type and number of operands
case "$OP" in
gen)
	if [ $# -ne 2 ]; then
		usage
	fi
	;;
enc|dec)
	if [ $# -ne 3 ]; then
		usage
	fi
	;;
*)
	usage
esac

# Create temporary working directory
D=`mktemp -d "${TMPDIR:-/tmp}/pkesh.XXXXXX"`
trap 'rm -r "$D"' EXIT

# Perform the operation
$OP "$@"
